//===============================================================================
// Microsoft patterns & practices Enterprise Library
// Configuration Application Block
//===============================================================================
// Copyright  Microsoft Corporation.  All rights reserved.
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY
// OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT
// LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
// FITNESS FOR A PARTICULAR PURPOSE.
//===============================================================================

using System;
using System.Reflection;
using Microsoft.Practices.EnterpriseLibrary.Configuration.Factory;
using Microsoft.Practices.EnterpriseLibrary.Configuration.Properties;

namespace Microsoft.Practices.EnterpriseLibrary.Configuration.Injection
{
	/// <summary>
	/// Indicates that the property value of the class is a name reference to a collection that can be created through the factory type by name.
	/// </summary>
	[AttributeUsage(AttributeTargets.Class, AllowMultiple = true, Inherited = false)]
	public sealed class ReferenceCollectionPropertyParameterFactoryAttribute : CollectionPropertyParameterFactoryAttribute
	{
		/// <summary>
		/// Initialize a new instance of the <see cref="ReferenceCollectionPropertyParameterAttribute"/> class with the order, the parameter type, the property name and the type of collection to create.
		/// </summary>
		/// <param name="order">The ordinal position of the constructor parameter.</param>
		/// <param name="parameterType">The parameter type.</param>
		/// <param name="propertyName">The name of the property to get the parameter value.</param>
		/// <param name="collectionType">The type of collection to create.</param>
		/// <param name="factoryType">The <see cref="INamedObjectFactory"/> used to create the collection.</param>
        public ReferenceCollectionPropertyParameterFactoryAttribute(Type collectionType, Type elementFactoryType)
            : base(collectionType, elementFactoryType)
		{
            if (elementFactoryType == null) throw new ArgumentNullException("elementFactoryType");
            if (!typeof(INamedObjectFactory<object, object>).IsAssignableFrom(elementFactoryType))
			{
				throw new ArgumentException(Resources.ExceptionTypeNotINamedObjectFactory, "factoryType");
			}
		}

		/// <summary>
		/// Gets the <see cref="ReferenceFactory"/> to use to transform the nested element.
		/// </summary>
        /// <returns>The <see cref="ReferenceFactory"/> to use to transform the nested element.</returns>
        protected override IObjectFactory<TObjectToCreate, TObjectConfiguration> GetElementFactory<TObjectToCreate, TObjectConfiguration>()
		{
            return new ReferenceFactory<TObjectToCreate, TObjectConfiguration>(base.GetElementFactory<TObjectToCreate, TObjectConfiguration>());
		}
	}
}
